//
//  Agora SDK
//
//  Copyright (c) 2020 Agora.io. All rights reserved.
//
#pragma once  // NOLINT(build/header_guard)

#include "AgoraBase.h"
#include "AgoraRefPtr.h"
#include "IAgoraMediaRtcRecorderEventHandler.h"


namespace agora {
namespace base {
class IAgoraService;
}
namespace rtc {

class IAgoraMediaRtcRecorderEventHandler;

/*
* The definition of UserMixerLayout.
*/
struct UserMixerLayout
{
  /*
  * The user ID of the user whose video stream you want to mix.
  */
  user_id_t userId;
  /*
  * The layout configuration of the user's video stream in the mixed video stream.
  */
  MixerLayoutConfig config;
};

/*
* The definition of VideoMixingLayout.
*/
struct VideoMixingLayout
{
  /*
  * The width of the mixed video stream.
  */
  int canvasWidth;
  /*
  * The height of the mixed video stream.
  */
  int canvasHeight;
  /*
  * The frame rate (fps) of the mixed video stream.
  */
  int canvasFps;
  /*
  * The background color of the mixed video stream.
  * The background color in RGB hex value. Value only. Do not include a preceeding #. For example,
  * 0xFFB6C1 (light pink). The default value is 0x000000 (black).
  */
  uint32_t backgroundColor;
  /*
  * The background image of the mixed video stream.
  * The path of the background image. The default value is NULL, which means no background image.
  * If you set this parameter, the background image must be in PNG or JPG format.
  */
  const char*  backgroundImage;
  /*
  * The number of users in the mixed video stream.
  */
  uint32_t userLayoutConfigNum;
  /*
  * The layout configurations of the users in the mixed video stream.
  */
  const UserMixerLayout* userLayoutConfigs; 
};

enum VideoFrameCaptureType {
  /** 1: Video frame as origin format from sender, i.e., H.264 or H.265.*/
  VIDEO_FORMAT_ENCODED_FRAME_TYPE = 0,
  /** 2: Video frame in YUV format. */
  VIDEO_FORMAT_YUV_FRAME_TYPE = 1,
  /** 3: Video frame in JPG format. */
  VIDEO_FORMAT_JPG_FRAME_TYPE = 2,
  /** 4: Video frame in JPG file format. */
  VIDEO_FORMAT_JPG_FILE_TYPE = 3,
};

struct RecorderVideoFrameCaptureConfig {
  /* The video frame capture type, see \ref agora::rtc::VideoFrameCaptureType "VideoFrameCaptureType".*/
  VideoFrameCaptureType videoFrameType;

  /**
  * Specifies the absolute path to the directory where JPG files will be saved.
  *
  * When `videoFrameType` is set to `VIDEO_FORMAT_JPG_FILE_TYPE` and `jpgFileStorePath` is not null,
  * the JPG file will be saved to this directory.
  * The filename will follow the format: "channelId_userId_YYYYMMDDHHMMSS_MS.jpg".
  *
  * @note Make sure that the specified directory already exists and is writable.
  */
  const char* jpgFileStorePath;
  
  /* The interval of the video frame capture for jpg in seconds.The default value is 5 seconds, min 1 seconds*/
  int jpgCaptureIntervalInSec;

  /* recorder video frame observer to get frame*/
  IRecorderVideoFrameObserver* observer;

  RecorderVideoFrameCaptureConfig():
    videoFrameType(VIDEO_FORMAT_ENCODED_FRAME_TYPE),
    jpgFileStorePath(NULL),
    jpgCaptureIntervalInSec(5),
    observer(NULL)
    {}
};

/**
 * The IAgoraMediaRtcRecorder provides functions to record media stream from rtc channel.
 * You can create it by IMediaComponentFactory::createMediaRtcRecorder().
 * If you want to record multiple media stream,create multiple  IAgoraMediaRtcRecorder objects.
 * Enjoy your recording experience。
 */
class IAgoraMediaRtcRecorder : public RefCountInterface{

protected:
  virtual ~IAgoraMediaRtcRecorder() {}

public:
  /** 
   *  Initialize the recorder
   *  @param agora_service The agora service,please make sure it has initialized.
   *  @param enable_mix  Whether to enable the mixing mode when recording:
   *                      - true: (Default) Enable.
   *                      - false: Do not enable.
   *  @param recordEncodedOnly Whether to record encoded video only:
   *                      - true: the received video will not be decoded, and the video data will be recorded in encoded format.
   *                      - false: （Default) the received video will be decoded and encoded.
   *  @note recordEncodedOnly is only valid when enable_mix is false. 
   *  @return
   *   - = 0: init success.
   *   - < 0: Failure.
   */
  virtual int initialize(base::IAgoraService* agora_service, bool enable_mix = true, bool recordEncodedOnly = false) = 0;

  /** 
   *  Join the Agora rtc channel.
   *  @param token The app ID or token.
   *  @param channelId The channel name. It must be in the string format and not exceed 64 bytes in length. Supported character scopes are:
   *  - All lowercase English letters: a to z.
   *  - All uppercase English letters: A to Z.
   *  - All numeric characters: 0 to 9.
   *  - The space character.
   *  - Punctuation characters and other symbols, including: "!", "#", "$", "%", "&", "(", ")", "+",
   *  "-", ":", ";", "<", "=",
   *  ".", ">", "?", "@", "[", "]", "^", "_", " {", "}", "|", "~", ","
   *  @param userId The ID of the local user. If you do not specify a user ID or set `userId` as `null`,
   *  @return
   *  - = 0: init success.
   *  - < 0: Failure.
   */
  virtual int joinChannel(const char* token,const char* channelId,const char* userId) = 0;

  /**
   * leave from the Agora rtc channel.
   * @return
   * - 0: Success.
   * - < 0: Failure.
   */
  virtual int leaveChannel() = 0;

  /** Enables/Disables the built-in encryption.
   *
   * In scenarios requiring high security, Agora recommends calling this method to enable the built-in encryption before joining a channel.
   *
   * All users in the same channel must use the same encryption mode and encryption key. Once all users leave the channel, the encryption key of this channel is automatically cleared.
   *
   *
   * @param enabled Whether to enable the built-in encryption:
   * - true: Enable the built-in encryption.
   * - false: (Default) Disable the built-in encryption.
   * @param config Configurations of built-in encryption schemas. See \ref agora::rtc::EncryptionConfig "EncryptionConfig".
   *
   * @return
   * - 0: Success.
   * - < 0: Failure.
   */
  virtual int enableEncryption(bool enabled, const EncryptionConfig& config) = 0;

  /**
   * Subscribes and record to the audio of all remote users in the channel.
   *
   * This method also automatically subscribes to the audio of any subsequent remote user.
   *
   * @return
   * - 0: Success.
   * - < 0: Failure.
   */
  virtual int subscribeAllAudio() = 0;

  /**
   * Subscribes to the video of all remote users in the channel.
   *
   * This method also automatically subscribes to the video of any subsequent remote user.
   *
   * @param subscriptionOptions The reference to the video subscription options: \ref agora::rtc::VideoSubscriptionOptions "VideoSubscriptionOptions".
   * @return
   * - 0: Success.
   * - < 0: Failure.
   */
  virtual int subscribeAllVideo(const agora::rtc::VideoSubscriptionOptions& subscriptionOptions) = 0;

  /**
   * Stops subscribing to the audio of all remote users in the channel.
   *
   * This method automatically stops subscribing to the audio of any subsequent user, unless you explicitly
   * call \ref subscribeAudio "subscribeAudio" or \ref subscribeAllAudio "subscribeAllAudio".
   *
   * @return
   * - 0: Success.
   * - < 0: Failure.
   */
  virtual int unsubscribeAllAudio() = 0;

  /**
   * Stops subscribing to the video of all remote users in the channel.
   *
   * This method automatically stops subscribing to the video of any subsequent user, unless you explicitly
   * call \ref subscribeVideo "subscribeVideo" or \ref subscribeAllVideo "subscribeAllVideo".
   *
   * @return
   * - 0: Success.
   * - < 0: Failure.
   */
  virtual int unsubscribeAllVideo() = 0;

  /**
   * Subscribes and record to the audio of a specified remote user in channel.
   *
   * @param userId The ID of the remote user whose audio you want to subscribe to.
   * @return
   * - 0: Success.
   * - < 0: Failure.
   *   - -2(ERR_INVALID_ARGUMENT), if no such user exists or `userId` is invalid.
   */
  virtual int subscribeAudio(user_id_t userId) = 0;
  
  /**
   * Stops subscribing to the audio of a specified remote user in the channel.
   *
   * @param userId The ID of the user whose audio you want to stop subscribing to.
   * @return
   * - 0: Success.
   * - < 0: Failure.
   *   - -2(ERR_INVALID_ARGUMENT), if no such user exists or `userId` is invalid.
   */
  virtual int unsubscribeAudio(user_id_t userId) = 0;

  /**
   * Subscribes to the video of a specified remote user in the channel.
   *
   * @param userId The ID of the user whose video you want to subscribe to.
   * @param subscriptionOptions The reference to the video subscription options: \ref agora::rtc::VideoSubscriptionOptions "VideoSubscriptionOptions".
   * For example, subscribing to encoded video data only or subscribing to low-stream video.
   *
   * @return
   * - 0: Success.
   * - < 0: Failure.
   *   - -2(ERR_INVALID_ARGUMENT), if no such user exists or `userId` is invalid.
   */
  virtual int subscribeVideo(user_id_t userId, const VideoSubscriptionOptions& subscriptionOptions) = 0;
    
  /**
   * Stops subscribing to the video of a specified remote user in the channel.
   *
   * @param userId The ID of the user whose video you want to stop subscribing to.
   * @return
   * - 0: Success.
   * - < 0: Failure.
   *   - -2(ERR_INVALID_ARGUMENT), if no such user exists or `userId` is invalid.
   */
  virtual int unsubscribeVideo(user_id_t userId) = 0;
  
  /**
   * Sets the time interval of the onAudioVolumeIndication callback.
   * @param intervalInMS Sets the time interval(ms) between two consecutive volume indications. The default
   * value is 500.
   * - &le; 10: Disables the volume indication.
   * - > 10: The time interval (ms) between two consecutive callbacks.
   *
   * @return
   * - 0: Success.
   * - < 0: Failure.
   */
  virtual int setAudioVolumeIndicationParameters(int intervalInMS) = 0;
  
  /**
   *  Sets the layout for the mixed video stream.
   *  @param layout The layout of the mixed video stream you want \ref agora::rtc::VideoMixingLayout
   *  @return
   *  - 0: Success.
   *  - < 0: Failure.
   */
  virtual int setVideoMixingLayout(const VideoMixingLayout& layout) = 0;

  /**
   * Set the recorder config
   * @param config The config of the recorder \ref agora::media::MediaRecorderConfiguration
   * @return
   *  - 0: Success.
   *  - < 0: Failure.
  */
  virtual int setRecorderConfig(const media::MediaRecorderConfiguration& config) = 0;

  /**
   * Add the watermark for the stream
   * @param watermark_configs The  watermark config
   * @param num The number of watermark configurations.
   * @return
   *  - 0: Success.
   *  - < 0: Failure.
  */
  virtual int enableAndUpdateVideoWatermarks(WatermarkConfig* watermark_configs, int num) = 0;

  /**
   * Disables the watermark for the stream
   * @return
   *  - 0: Success.
   *  - < 0: Failure.
   */
  virtual int disableVideoWatermarks() = 0;
  /**
   * Start the recording 
   * @return
   *  - 0: Success.
   *  - < 0: Failure.
  */
  virtual int startRecording() = 0;

  /**
   * Stop the recording 
   * @return
   *  - 0: Success.
   *  - < 0: Failure.
  */
  virtual int stopRecording() = 0;

  /**
   * Start the recording by userid
   * @param userId The ID of the user whose stream you want to start recorde to.
   * @return
   *  - 0: Success.
   *  - < 0: Failure.
  */
  virtual int startSingleRecordingByUid(user_id_t userId) = 0;

  /**
   * Stop the recording by userid
   * @param userId The ID of the user whose stream you want to stop recorde to.
   * @return
   *  - 0: Success.
   *  - < 0: Failure.
  */
  virtual int stopSingleRecordingByUid(user_id_t userId) = 0;

  /**
   * Sets the recording config by userid
   * @param config The config of the recorder \ref agora::media::MediaRecorderConfiguration
   * @param userId The ID of the user whose stream you want to stop recorde to.
   * @return
   *  - 0: Success.
   *  - < 0: Failure.
  */
  virtual int setRecorderConfigByUid(const media::MediaRecorderConfiguration& config, user_id_t userId) = 0;

   /**
   * Add the watermark for the stream by uid
   * @param watermark_configs The  watermark config
   * @param num The  watermark config nums;
   * @param userId The ID of the user whose stream you want to add watermark.
   * @return
   *  - 0: Success.
   *  - < 0: Failure.
  */
  virtual int enableAndUpdateVideoWatermarksByUid(WatermarkConfig* watermark_configs, int num, user_id_t userId) = 0;

  /**
   * disable the watermark for the stream by uid
   * @param userId The ID of the user whose stream you want to disable watermark.
   * @return
   *  - 0: Success.
   *  - < 0: Failure.
   */
  virtual int disableVideoWatermarksByUid(user_id_t userId) = 0;

  /**
   * Registers the event handler for recording events.
   * @param handler The handler of the recorder \ref agora::rtc::IAgoraMediaRtcRecorderEventHandler
   * @return
   *  - 0: Success.
   *  - < 0: Failure.
  */
  virtual int registerRecorderEventHandle(IAgoraMediaRtcRecorderEventHandler* handler) = 0;

  /**
   * Unregister the recording event handle
   * @param handler The handler of the recorder \ref agora::rtc::IAgoraMediaRtcRecorderEventHandler
   * @return
   *  - 0: Success.
   *  - < 0: Failure.
  */
  virtual int unregisterRecorderEventHandle(IAgoraMediaRtcRecorderEventHandler* handler) = 0;

   /** 
    * Set Global local access point addresses in local Ap mode(which also calls local proxy).
    * Use this method before join channel.
    * This method will influence all the recorder instances in the same process, you only need to call it once.
    * @param config The LocalAccessPointConfiguration class, See the definition of
    * LocalAccessPointConfiguration for details.
    * @return
    *  - 0: Success
    *  - < 0: Failure
  */
  virtual int setGlobalLocalAccessPoint(const LocalAccessPointConfiguration& config) = 0;

  /**
   * Renews the token.
   *
   * The token expires after a certain period of time. Under the following circumstances, you 
   * must generate a new token from the server and then call this method to renew it. 
   * Otherwise, the SDK disconnects from the Agora channel:
   * - The \ref IAgoraMediaRtcRecorderEventHandler::onError "onError" callback reports `ERR_TOKEN_EXPIRED(109)`
   * - The \ref IAgoraMediaRtcRecorderEventHandler::onTokenPrivilegeWillExpire "onTokenPrivilegeWillExpire" callback is triggered
   * - The \ref IAgoraMediaRtcRecorderEventHandler::onTokenPrivilegeDidExpire "onTokenPrivilegeDidExpire" callback is triggered
   *
   * @param token The pointer to the new token.
   * @return
    *  - 0: Success
    *  - < 0: Failure
   */
  virtual int renewToken(const char* token) = 0;

  /*
  * configure the recorder video frame capture
  *
  * @param enable Whether to enable the video frame capture
  * @param config The config of the recorder video frame capture \ref agora::rtc::RecorderVideoFrameCaptureConfig
  * @return
  *  - 0: Success.
  *  - < 0: Failure.
  */
  virtual int enableRecorderVideoFrameCapture(bool enable, const RecorderVideoFrameCaptureConfig& config) = 0;
};

}//namespace rtc
}// namespace agora