#include "agoraTest.h"

#include <time.h>
#include <unistd.h>

#include "common/file_parser/helper_h264_parser.h"
#include "common/helper.h"
#include "common/sample_common.h"

int agoraTest::setConfiguration() {
  connectionConfig_.autoSubscribeAudio = false;
  connectionConfig_.autoSubscribeVideo = false;
  connectionConfig_.clientRoleType = agora::rtc::CLIENT_ROLE_BROADCASTER;
  encoderConfig_.codecType = agora::rtc::VIDEO_CODEC_H264;
  encoderConfig_.dimensions.width = 352;
  encoderConfig_.dimensions.height = 288;
  encoderConfig_.frameRate = 15;
  return 0;
}

int agoraTest::initializeAndconnect(const char* Channel) {
  connection_ = Service_->createRtcConnection(connectionConfig_);
  if (!connection_) {
    AG_LOG(ERROR, "Failed to creating Agora connection!");
    return -1;
  }
  connObserver_ = std::make_shared<SampleConnectionObserver>();
  connection_->registerObserver(connObserver_.get());

  if (connection_->connect(Appid, Channel, nullptr, 0)) {
    AG_LOG(ERROR, "Failed to connect to Agora channel!");
    return -1;
  }

  localUserObserver_ = std::make_shared<SampleLocalUserObserver>(connection_->getLocalUser());
  pcmFrameObserver_ = std::make_shared<PcmFrameObserver>();
  if (connection_->getLocalUser()->setPlaybackAudioFrameBeforeMixingParameters(1, 48000)) {
    AG_LOG(ERROR, "Failed to set audio frame parameters!");
    return -1;
  }
  localUserObserver_->setAudioFrameObserver(pcmFrameObserver_.get());
  h264FrameReceiver_ = std::make_shared<H264FrameReceiver>();
  localUserObserver_->setVideoEncodedImageReceiver(h264FrameReceiver_.get());
  yuvFrameObserver_ = std::make_shared<YuvFrameObserver>();
  localUserObserver_->setVideoFrameObserver(yuvFrameObserver_.get());

  factory_ = Service_->createMediaNodeFactory();
  if (!factory_) {
    AG_LOG(ERROR, "Failed to create media node factory!");
  }
  audioFrameSender_ = factory_->createAudioPcmDataSender();
  if (!audioFrameSender_) {
    AG_LOG(ERROR, "Failed to create audio data sender!");
    return -1;
  }
  customAudioTrack_ = Service_->createCustomAudioTrack(audioFrameSender_);
  if (!customAudioTrack_) {
    AG_LOG(ERROR, "Failed to create audio track!");
    return -1;
  }
  videoEncodedFrameSender_ = factory_->createVideoEncodedImageSender();
  if (!videoEncodedFrameSender_) {
    AG_LOG(ERROR, "Failed to create video frame sender!");
    return -1;
  }
  agora::rtc::SenderOptions option;
  option.ccMode = agora::rtc::TCcMode::CC_ENABLED;
  // Create video track
  customEncodedVideoTrack_ = Service_->createCustomVideoTrack(videoEncodedFrameSender_, option);
  if (!customEncodedVideoTrack_) {
    AG_LOG(ERROR, "Failed to create video track!");
    return -1;
  }
  videoFrameSender_ = factory_->createVideoFrameSender();
  if (!videoFrameSender_) {
    AG_LOG(ERROR, "Failed to create video frame sender!");
    return -1;
  }
  customVideoTrack_ = Service_->createCustomVideoTrack(videoFrameSender_);
  if (!customVideoTrack_) {
    AG_LOG(ERROR, "Failed to create video track!");
    return -1;
  }
  customVideoTrack_->setVideoEncoderConfiguration(encoderConfig_);
  customVideoTrack_->setEnabled(true);
  customAudioTrack_->setEnabled(true);
  customEncodedVideoTrack_->setEnabled(true);
  // connection_->getLocalUser()->publishVideo(customEncodedVideoTrack_);
  // connection_->getLocalUser()->publishVideo(customVideoTrack_);
  // // Wait until connected before sending media stream
  connObserver_->waitUntilConnected(DEFAULT_CONNECT_TIMEOUT_MS);
  AG_LOG(INFO, "initializeAndconnect successful!");
  return 0;
}

int agoraTest::release() {
  connection_->getLocalUser()->unpublishAudio(customAudioTrack_);
  connection_->getLocalUser()->unpublishVideo(customEncodedVideoTrack_);
  connection_->getLocalUser()->unpublishVideo(customVideoTrack_);
  connection_->unregisterObserver(connObserver_.get());
  connObserver_.reset();
  audioFrameSender_ = nullptr;
  videoFrameSender_ = nullptr;
  videoEncodedFrameSender_ = nullptr;
  customAudioTrack_ = nullptr;
  customVideoTrack_ = nullptr;
  customEncodedVideoTrack_ = nullptr;
  factory_ = nullptr;
  localUserObserver_->unsetAudioFrameObserver();
  localUserObserver_->unsetVideoFrameObserver();
  localUserObserver_.reset();
  pcmFrameObserver_.reset();
  h264FrameReceiver_.reset();
  yuvFrameObserver_ = nullptr;
  connection_ = nullptr;
  AG_LOG(INFO, "release successful!");
  return 0;
}

int agoraTest::sendH264Task(const char* videoFile, int frameRate, bool& exitFlag) {
  connection_->getLocalUser()->publishVideo(customEncodedVideoTrack_);

  std::unique_ptr<HelperH264FileParser> h264FileParser(new HelperH264FileParser(videoFile));
  h264FileParser->initialize();

  // Calculate send interval based on frame rate. H264 frames are sent at this
  // interval
  PacerInfo pacer = {0, 1000 / frameRate, 0, std::chrono::steady_clock::now()};

  while (!exitFlag && Timer.CheckTime()) {
    if (auto h264Frame = h264FileParser->getH264Frame()) {
      agora::rtc::EncodedVideoFrameInfo videoEncodedFrameInfo;
      videoEncodedFrameInfo.rotation = agora::rtc::VIDEO_ORIENTATION_0;
      videoEncodedFrameInfo.codecType = agora::rtc::VIDEO_CODEC_H264;
      videoEncodedFrameInfo.framesPerSecond = frameRate;
      videoEncodedFrameInfo.frameType =
          (h264Frame.get()->isKeyFrame
               ? agora::rtc::VIDEO_FRAME_TYPE::VIDEO_FRAME_TYPE_KEY_FRAME
               : agora::rtc::VIDEO_FRAME_TYPE::VIDEO_FRAME_TYPE_DELTA_FRAME);
      videoEncodedFrameSender_->sendEncodedVideoImage(
          reinterpret_cast<uint8_t*>(h264Frame.get()->buffer.get()), h264Frame.get()->bufferLen,
          videoEncodedFrameInfo);
      waitBeforeNextSend(pacer);  // sleep for a while before sending next frame
    }
  };
  return 0;
}

int agoraTest::sendPcmTask(const char* audioFile, int numOfChannels, int sampleRate,
                           bool& exitFlag) {
  PacerInfo pacer = {0, 10, 0, std::chrono::steady_clock::now()};
  connection_->getLocalUser()->publishAudio(customAudioTrack_);
  FILE* file = nullptr;
  while (!exitFlag && Timer.CheckTime()) {
    // Calculate byte size for 10ms audio samples
    int sampleSize = sizeof(int16_t) * numOfChannels;
    int samplesPer10ms = sampleRate / 100;
    int sendBytes = sampleSize * samplesPer10ms;

    if (!file) {
      if (!(file = fopen(audioFile, "rb"))) {
        AG_LOG(ERROR, "Failed to open audio file %s", audioFile);
        return -1;
      }
      AG_LOG(INFO, "Open audio file %s successfully", audioFile);
    }

    uint8_t frameBuf[sendBytes];

    if (fread(frameBuf, 1, sizeof(frameBuf), file) != sizeof(frameBuf)) {
      if (feof(file)) {
        fclose(file);
        file = nullptr;
        AG_LOG(INFO, "End of audio file");
      } else {
        AG_LOG(ERROR, "Error reading audio data: ");
      }
      return 0;
    }

    if (audioFrameSender_->sendAudioPcmData(frameBuf, 0, 0, samplesPer10ms,
                                            agora::rtc::TWO_BYTES_PER_SAMPLE, numOfChannels,
                                            sampleRate) < 0) {
      AG_LOG(ERROR, "Failed to send audio frame!");
    }
    waitBeforeNextSend(pacer);  // sleep for a while before sending next frame
  }
  if (file) fclose(file);
  return 0;
}

int agoraTest::sendYuvTask(const char* videoFile, int frameRate, int width, int height,
                           bool& exitFlag) {
  PacerInfo pacer = {0, 1000 / frameRate, 0, std::chrono::steady_clock::now()};
  connection_->getLocalUser()->publishVideo(customVideoTrack_);
  while (!exitFlag && Timer.CheckTime()) {
    static FILE* file = nullptr;
    // Calculate byte size for YUV420 image
    int sendBytes = width * height * 3 / 2;

    if (!file) {
      if (!(file = fopen(videoFile, "rb"))) {
        AG_LOG(ERROR, "Failed to open video file %s", videoFile);
        return -1;
      }
      AG_LOG(INFO, "Open video file %s successfully", videoFile);
    }

    uint8_t frameBuf[sendBytes];

    if (fread(frameBuf, 1, sizeof(frameBuf), file) != sizeof(frameBuf)) {
      if (feof(file)) {
        fclose(file);
        file = nullptr;
        AG_LOG(INFO, "End of video file");
      } else {
        AG_LOG(ERROR, "Error reading video data: %s", std::strerror(errno));
      }
    }
    agora::media::base::ExternalVideoFrame videoFrame;
    videoFrame.type = agora::media::base::ExternalVideoFrame::VIDEO_BUFFER_RAW_DATA;
    videoFrame.format = agora::media::base::VIDEO_PIXEL_I420;
    videoFrame.buffer = frameBuf;
    videoFrame.stride = width;
    videoFrame.height = height;
    videoFrame.cropLeft = 0;
    videoFrame.cropTop = 0;
    videoFrame.cropRight = 0;
    videoFrame.cropBottom = 0;
    videoFrame.rotation = 0;
    videoFrame.timestamp = 0;

    if (videoFrameSender_->sendVideoFrame(videoFrame) < 0) {
      AG_LOG(ERROR, "Failed to send video frame!");
    }
    // AG_LOG(INFO, "send successfully");

    waitBeforeNextSend(pacer);  // sleep for a while before sending next frame
  }
  return 0;
}
